/* ***************************************************************** 
   MESQUITE -- The Mesh Quality Improvement Toolkit

   Copyright 2004 Sandia Corporation and Argonne National
   Laboratory.  Under the terms of Contract DE-AC04-94AL85000 
   with Sandia Corporation, the U.S. Government retains certain 
   rights in this software.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License 
   (lgpl.txt) along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
   diachin2@llnl.gov, djmelan@sandia.gov, mbrewer@sandia.gov, 
   pknupp@sandia.gov, tleurent@mcs.anl.gov, tmunson@mcs.anl.gov,
   kraftche@cae.wisc.edu         
   
   ***************************************************************** */
/*!
  \file   MeshBoundaryDomain2D.hpp
  \brief  


  \author Brian Miller
  \date   2010-10-10
*/
#ifndef MSQ_MESH_BOUNDARY_DOMAIN_2D_HPP
#define MSQ_MESH_BOUNDARY_DOMAIN_2D_HPP

#include "MeshInterface.hpp"
#include "Vector3D.hpp"
#include "MsqVertex.hpp"
#include <set>

namespace MESQUITE_NS
{
typedef struct MyVertex
{
  MsqVertex coordinate;
  MsqVertex snapped_to_position;
  int snapped_onto_edge;
  int edges;
  int current_edge;
  Vector3D normal;
  int other_edge;
  Vector3D other_normal;
  MyVertex* closest;
} MyVertex;

/*! \class MeshBoundaryDomain2D
  It will provide the normal information necessary for surface mesh optimization.
*/
class MeshBoundaryDomain2D : public MeshDomain
{
public:
  enum Plane{ XY=2, XZ=1, YZ=0};
  // mine

  enum InterfaceSnapType { LINEAR, QUADRATIC, MOMENTPRESERVING };
    
  MeshBoundaryDomain2D(Plane orient, double offset=0.0, bool project_gradient = true, InterfaceSnapType theSnapType=LINEAR );
  void skin_area_mesh(Mesh* mesh, float cos_crease_angle=0.9f, const char* material_tag_name=0);
  void copy_surface_mesh(Mesh* mesh);

  void get_boundary_vertices( std::vector<Mesh::VertexHandle> &theBoundaryVertices )
  {
    theBoundaryVertices = boundary_vertices;
  };
    
  void get_boundary_edges( std::vector<Mesh::VertexHandle> &theBoundaryEdges ) 
  {
    theBoundaryEdges = boundary_edges;
  };
    
  void get_corner_vertices( std::vector<Mesh::VertexHandle> &theCornerVertices ) 
  {
    theCornerVertices.clear();
    std::copy(corner_vertices.begin(), corner_vertices.end(), back_inserter(theCornerVertices));
  };
    
  // inherited

  virtual ~MeshBoundaryDomain2D();
    
  virtual void snap_to(Mesh::VertexHandle entity_handle,
		       Vector3D &coordinate) const;
    
  virtual void vertex_normal_at(Mesh::VertexHandle entity_handle,
				Vector3D &coordinate) const;
     
  virtual void element_normal_at(Mesh::ElementHandle entity_handle,
				 Vector3D &coordinate) const;
   
  virtual void vertex_normal_at(const Mesh::VertexHandle* handle,
				Vector3D coords[],
				unsigned count,
				MsqError& err) const;

  virtual void closest_point( Mesh::VertexHandle handle,
			      const Vector3D& position,
			      Vector3D& closest,
			      Vector3D& normal,
			      MsqError& err ) const;

  virtual void domain_DoF( const Mesh::VertexHandle* handle_array,
			   unsigned short* dof_array,
			   size_t num_vertices,
			   MsqError& err ) const;

private:

  // Simpler quadraic snap
  void quadraticSnap(double *xs, double *ys, double x[3], double y[3]) const;
    
  // Moment preserving snapping algorithm
  void conservedFitting(double *xs, double *ys, double x[4], double y[4]);

  void QuadraticSolver(double *c);

  void CholeskyInverse(double **a, double *b, int rank);
  // the 2D mesh 
  Mesh* mesh;

  //
  // mNormal is the normal to the plane this 2D mesh lies in and mCoeff handles offset from the origin
  Vector3D mNormal;
  double mCoeff;

  // handle to hold the MyVertex data on the boundary vertices
  TagHandle mesh_boundary_vertex_tag;

  // 1-1 mapping of boundary_edges vertex to MyVertex*.  For the
  // vertex handle at boundary_edges[i], boundary_mesh[i] will hold 
  // the MyVertex* for that vertex. 
  std::vector<MyVertex*> boundary_mesh;

  // 
  // for each boundary_edge, one outward pointing normal is stored
  // in edge_normals
  std::vector<Vector3D> edge_normals;
  bool project_gradient;

  // list of unique boundary vertex handles
  std::vector<Mesh::VertexHandle> boundary_vertices;

  // set of corner vertices (std::set used to prevent potential duplicates)
  std::set<Mesh::VertexHandle> corner_vertices;

  // consectutive pairs define boundary edges, given in correct order for
  // orientation purposes (interior of domain is to the left).  Note that 
  // since a vertex is held by two (or more) edges there are duplicate 
  // VertexHandles in boundary_edges.  
  std::vector<Mesh::VertexHandle> boundary_edges;

  InterfaceSnapType mInterfaceSnapType;
  double PICO;
  double MEGA;
  double MICRO;
  int ITER_MAX;
};
}

#endif
